//*************************************************************************************************
//
//	Description:
//		foliage_treewall.fx - Shader for non-instanced rendering of treewalls.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship			07/14/2007	0.1						Initial branch from foliage.fx
//	<TABLE>
//
//*************************************************************************************************
#define _SSAO_READY_
// Yet another preprocessor hack to special-case the lighting in this shader
#define _USE_HALF_LAMBERTIAN_LIGHTING_

// Tell the lighting backend we want transmissive calcs
// (N.B. This feels messy. Find a better way to do it.)
#define USE_TRANSMISSIVE_LIGHTING

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#define USE_SPECULAR
#define CONTROL_MAPPING
#define ALPHA_FRESNEL
#endif



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
>;

#endif


//
// Textures
//

SPECIALISATION_PARAM_DEFAULTS_TRUE( alphaFresnel, "Alpha Fresnel?", "ALPHA_FRESNEL" )

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

SPECIALISATION_PARAM( useSpecular, "Use specular?", "USE_SPECULAR" )	// TRUE if the specular lighting is to be used

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_TEXTURE_PARAM( specularTexture, "Spec Tex {UV1}", useSpecular )							// Specular colour in RGB, shininess in alpha
#endif

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( minSpecPower, "Min Specular Power", useSpecular, 1.0f, 1024.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( maxSpecPower, "Max Specular Power", useSpecular, 1.0f, 1024.0f, 32.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalSpecularFactor, "Specular Factor", useSpecular, 0.0f, 10.0f, 1.0f )
#endif


// Control map
SPECIALISATION_PARAM( useControlMap, "Use control map?", "CONTROL_MAPPING" )	// TRUE if a control map is to be used for extra effects

#if defined( _3DSMAX_ ) || defined( CONTROL_MAPPING )
DEPENDENT_TEXTURE_PARAM( controlTexture, "Control Tex {UV1}", useControlMap )
#endif

float4 transmissiveMult
<
	string UIName = "Trans Mult";
	string UIWidget = "Spinner";
	float UIMin = 0.0f;
	float UIMax = 16.0f;
	bool appEdit = true;
	bool export = true;
> = { 0.8f, 1.0f, 0.2f, 1.0f };


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( CONTROL_MAPPING )
sampler2D controlMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="controlTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < controlTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float3 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD2;													// Vertex alpha
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
	float3 normal   : NORMAL;															// Object space normal
};


struct SHADOWGEN_VSINPUT
{
	float3 position : POSITION;														// Object space position
	float2 texCoord : TEXCOORD0;													// Texture coords
};

// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 4 )
};



struct SHADOWGEN_VSOUTPUT
{
	float4 position			: POSITION;
	float2 texCoord			: TEXCOORD0;
#if defined(NEEDS_SHADOW_COORDS)
	float4 shadowCoord	: TEXCOORD1;
#endif
};

struct ZPRIMEDOF_VSOUTPUT
{
	float4 position			: POSITION;
	float2 texCoord			: TEXCOORD0;
	float4 coords				: TEXCOORD1;
};


//-----------------------------------------------------------------------
//
// Functions
//



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT TreewallVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif

	_output.texCoord = _input.texCoord;

	//
	// Apply wind animation to vertex pos if required
	//
	float4 localPosition = float4( _input.position, 1.0f );

	// Calculate clip-space position of the vertex
	_output.position = mul( localPosition, worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( localPosition, world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);
	float3 normalisedEyeVec = normalize( worldEyeVec );
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );

	DEPENDENT_CODE_START( alphaFresnel )
#if defined( _3DSMAX_ ) || defined( ALPHA_FRESNEL )
		float alphaFresnelScalar = abs( dot( normal, normalisedEyeVec ) );
		_output.colour.a *= sqrt( alphaFresnelScalar );
#endif
	DEPENDENT_CODE_END( alphaFresnel )

	// We just point the normal towards the eye
	normal = normalisedEyeVec;
	_output.normal = normal;

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
	float globalSpecularFactorValue = 0.0f;
	float minSpecPowerValue = 0.0f;
	float maxSpecPowerValue = 0.0f;
#else
	DEPENDENT_CODE_START( useSpecular )
	float globalSpecularFactorValue = globalSpecularFactor;
	DEPENDENT_CODE_END( useSpecular )
#endif

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}

SHADOWGEN_VSOUTPUT TreewallShadowGenVertexShader( SHADOWGEN_VSINPUT _input )
{
	SHADOWGEN_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;

	float4 localPosition = float4( _input.position, 1.0f );

	// Calculate clip-space position of the vertex
	_output.position = mul( localPosition, worldviewproj );

	// Clamp geometry that is behind the camera to the near plane so that it still renders.
	// We use an orthogonal projection, so doing this will not distort the shadow caster.
	CLAMP_SHADOW_Z;

#if defined(NEEDS_SHADOW_COORDS)
	OUTPUT_SHADOW_COORDS;
#endif	

	return _output;
}

ZPRIMEDOF_VSOUTPUT TreewallZPrimeDOFVertexShader( SHADOWGEN_VSINPUT _input )
{
	ZPRIMEDOF_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;

	float4 localPosition = float4( _input.position, 1.0f );

	// Calculate clip-space position of the vertex
	_output.position = mul( localPosition, worldviewproj );
	_output.coords = _output.position;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 4 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD2_centroid;								// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 4 )
	DECLARE_SHADOW_PS_INPUTS
};

#endif


struct SHADOWGEN_PSINPUT
{
	float2 texCoord			: TEXCOORD0;
#if defined(NEEDS_SHADOW_COORDS)
	float4 shadowCoord	: TEXCOORD1;
#endif	
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT TreewallFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	float4 diffuseTexColour;
	float4 specularTexColour;
	float4 transmissiveTexColour;

	//
	// Read control map (or substitute defaults if no control map used)
	//

	float4 controlTexColour;
	DEPENDENT_CODE_START( useControlMap )
#if defined( _3DSMAX_ ) || defined( CONTROL_MAPPING )
		controlTexColour = tex2D( controlMap, _input.texCoord );
#endif
	DEPENDENT_CODE_ELSE( useControlMap )
#if defined( _3DSMAX_ ) || !defined( CONTROL_MAPPING )
	controlTexColour = float4( 0.5f, 0.5f, 0.0f, 1.0f );		// 50% transmissive, average depth
#endif
	DEPENDENT_CODE_END( useControlMap )

	// Read diffuse colour
	diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	//
	// Calculate transmissive tex colour
	//
	float transmissiveness = controlTexColour.r;
	transmissiveTexColour = transmissiveMult * diffuseTexColour.g;

	float globalSpecularFactorValue;
	float minSpecPowerValue;
	float maxSpecPowerValue;

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
#else
	DEPENDENT_CODE_START( useSpecular )
	globalSpecularFactorValue = globalSpecularFactor;
	minSpecPowerValue = minSpecPower;
	maxSpecPowerValue = maxSpecPower;
	DEPENDENT_CODE_END( useSpecular )
#endif

	DEPENDENT_CODE_START( useSpecular )
#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
		// Read specular texture
		specularTexColour = tex2D( specularMap, _input.texCoord );
#endif
	DEPENDENT_CODE_ELSE( useSpecular )
#if defined( _3DSMAX_ ) || !defined( USE_SPECULAR )
		// No specular, so default the colour to zeros (will be optimised out)
		specularTexColour = float4( 0.0f, 0.0f, 0.0f, 0.0f );

		globalSpecularFactorValue = 0.0f;
		minSpecPowerValue = 0.0f;
		maxSpecPowerValue = 0.0f;
#endif
	DEPENDENT_CODE_END( useSpecular )

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// No normal map, so use interpolated normal
	normal = TSnormal;

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	// Copy the alpha over & rescale it to make maximum use of available accuracy
	accumulatedColour.w = saturate( diffuseTexColour.w / ( 128.0f / 255.0f ) );

	_output.Colour = CalculateOutputPixel( accumulatedColour );
	
	return _output;
}


PSOUTPUT TreewallShadowGenFragmentShader( SHADOWGEN_PSINPUT _input )
{
#if !defined(NEEDS_SHADOW_COORDS)
	PSOUTPUT output;
	
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	output.Colour = 0;
	output.Colour.a = saturate( diffuseTexColour.a / ( 128.0f / 255.0f ) );
	
	return output;
#else
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	CALC_SHADOWMAP_DEPTH( output.Colour, _input.shadowCoord );
	output.Colour.a = saturate( diffuseTexColour.a / ( 128.0f / 255.0f ) );
	SHADOWMAP_PS_ALPHATEST( output.Colour.a, 0.5f );

	return output;
#endif	
}

PSOUTPUT TreewallZPrimeDOFFragmentShader( ZPRIMEDOF_VSOUTPUT _input )
{
	PSOUTPUT output;
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	output.Colour = _input.coords.z / _input.coords.w;
	output.Colour.a = saturate( diffuseTexColour.a / ( 128.0f / 255.0f ) );

	return output;
}



//
// Low Detail Shaders
//

struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD1;														// Vertex colour
	float2 texCoord		: TEXCOORD0;											// UV coords for texture channel 0
};

VSOUTPUT_LD TreewallLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	float3 normalisedEyeVec = normalize( worldEyeVec );

	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	DEPENDENT_CODE_START( alphaFresnel )
#if defined( _3DSMAX_ ) || defined( ALPHA_FRESNEL )
		float alphaFresnelScalar = abs( dot( normal, normalisedEyeVec ) );
		_output.colour.a *= sqrt( alphaFresnelScalar );
#endif
	DEPENDENT_CODE_END( alphaFresnel )

	// We just point the normal towards the eye
	normal = normalisedEyeVec;

	// Do vertex lighting
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	return _output;
}

PSOUTPUT TreewallLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	// Apply vertex lighting to base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Copy the alpha over & rescale it to make maximum use of available accuracy
	accumulatedColour.w = saturate( diffuseTexColour.w / ( 128.0f / 255.0f ) );

	_output.Colour = CalculateLowDetailOutputPixel( accumulatedColour );
	
	return _output;
}




//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Treewall
<
	bool supportsSpecialisedLighting = true;
  bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Treewall";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour = "ERMB_RENDER";
	string zprimeDOFTechnique = "_Treewall_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour = "ERMB_RENDER";
	string shadowGenTechnique = "_Treewall_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string lowDetailBehaviour = "ERMB_RENDER";
	string lowDetailTechnique = "_Treewall_LowDetail";
	int    lowDetailDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
		bool	AlphaTestEnable = true;
		int		AlphaRef = 128;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = true;
		AlphaTestEnable = true;
		AlphaRef = 128;
		AlphaFunc = GreaterEqual;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx TreewallVertexShader();
		PixelShader = compile sce_fp_rsx TreewallFragmentShader();
#else
		VertexShader = compile vs_3_0 TreewallVertexShader();
		PixelShader = compile ps_3_0 TreewallFragmentShader();
#endif
	}
}


technique _Treewall_ShadowGen
{
	pass Pass0
	{
			SHADOWMAP_STATES_ALPHATEST( 128 )
#if defined (_PS3_)
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx TreewallShadowGenVertexShader();
			PixelShader = compile sce_fp_rsx TreewallShadowGenFragmentShader();
#elif defined(_XBOX)
			VertexShader = compile vs_3_0 TreewallShadowGenVertexShader();
			PixelShader = compile ps_3_0 TreewallShadowGenFragmentShader();
#else
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 TreewallShadowGenVertexShader();
			PixelShader = compile ps_3_0 TreewallShadowGenFragmentShader();
#endif
	}
}

technique _Treewall_ZPrime_DOF
{
	pass Pass0
	{
#if defined (_PS3_)
			AlphaTestEnable = true;
		  AlphaFunc = int2(GEqual, 128);
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx TreewallZPrimeDOFVertexShader();
			PixelShader = compile sce_fp_rsx TreewallZPrimeDOFFragmentShader();
#else
			AlphaTestEnable = true;
			AlphaRef = 128;
	    AlphaFunc = GreaterEqual;
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 TreewallZPrimeDOFVertexShader();
			PixelShader = compile ps_3_0 TreewallZPrimeDOFFragmentShader();
#endif
	}
}


technique _Treewall_LowDetail
<
  bool preservesGlobalState = true;
>
{
	pass Pass0
	{
#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx TreewallLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx TreewallLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 TreewallLowDetailVertexShader();
		PixelShader = compile ps_3_0 TreewallLowDetailFragmentShader();
#endif
	}
}
